<?php
/**
 * Hash Generator - Generate 512-character hash for protected pages
 * 
 * Usage: Run this script to generate a hash that can be used in URLs
 * URL format: domain.com/hash512/ or domain.com/protected-page.php?hash=hash512
 */

require_once 'detect-ip.php';

/**
 * Generate a 512-character hash
 * 
 * @return string 512-character hash
 */
function generateHash512() {
    // Generate random bytes and hash multiple times to get 512 characters
    $randomString = uniqid('', true) . microtime(true) . mt_rand() . $_SERVER['HTTP_USER_AGENT'];
    $hash = hash('sha256', $randomString);
    
    // Combine multiple hashes to reach 512 characters
    while (strlen($hash) < 512) {
        $hash .= hash('sha256', $hash . time() . mt_rand());
    }
    
    // Trim to exactly 512 characters
    return substr($hash, 0, 512);
}

// Generate hash
$hash = generateHash512();
$ip = detect_ip();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hash Generator</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 900px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 2px solid #28a745;
            padding-bottom: 10px;
        }
        .hash-box {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
            border: 2px solid #28a745;
            word-break: break-all;
            font-family: monospace;
            font-size: 12px;
        }
        .info {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border-left: 4px solid #007bff;
        }
        .url-examples {
            background: #fff3cd;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border-left: 4px solid #ffc107;
        }
        .url-examples code {
            display: block;
            background: #f8f9fa;
            padding: 10px;
            margin: 10px 0;
            border-radius: 3px;
            word-break: break-all;
        }
        button {
            background: #28a745;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 10px 5px;
        }
        button:hover {
            background: #218838;
        }
        .copy-btn {
            background: #007bff;
        }
        .copy-btn:hover {
            background: #0056b3;
        }
    </style>
    <script>
        function copyHash() {
            const hash = document.getElementById('hash').textContent;
            navigator.clipboard.writeText(hash).then(function() {
                alert('Hash copied to clipboard!');
            }, function() {
                alert('Failed to copy. Please select and copy manually.');
            });
        }
        
        function copyURL(format) {
            const hash = document.getElementById('hash').textContent;
            const domain = window.location.origin;
            let url;
            
            if (format === 'query') {
                url = domain + '/protected-page.php?hash=' + hash;
            } else {
                url = domain + '/' + hash + '/';
            }
            
            navigator.clipboard.writeText(url).then(function() {
                alert('URL copied to clipboard!');
            }, function() {
                alert('Failed to copy. Please select and copy manually.');
            });
        }
    </script>
</head>
<body>
    <div class="container">
        <h1>🔐 512-Character Hash Generator</h1>
        
        <div class="info">
            <h3>Generated Hash (512 characters)</h3>
            <div class="hash-box" id="hash"><?php echo htmlspecialchars($hash); ?></div>
            <button class="copy-btn" onclick="copyHash()">📋 Copy Hash</button>
        </div>
        
        <div class="url-examples">
            <h3>📝 URL Examples</h3>
            <p><strong>Option 1: Query Parameter (Works immediately)</strong></p>
            <code><?php echo $_SERVER['HTTP_HOST']; ?>/protected-page.php?hash=<?php echo htmlspecialchars($hash); ?></code>
            <button onclick="copyURL('query')">📋 Copy This URL</button>
            
            <p style="margin-top: 20px;"><strong>Option 2: URL Path (Requires .htaccess configuration)</strong></p>
            <code><?php echo $_SERVER['HTTP_HOST']; ?>/<?php echo htmlspecialchars($hash); ?>/</code>
            <button onclick="copyURL('path')">📋 Copy This URL</button>
        </div>
        
        <div class="info">
            <h3>ℹ️ Information</h3>
            <p><strong>Hash Length:</strong> <?php echo strlen($hash); ?> characters</p>
            <p><strong>Generated At:</strong> <?php echo date('Y-m-d H:i:s'); ?></p>
            <p><strong>Your IP:</strong> <?php echo htmlspecialchars($ip); ?></p>
            <p><strong>Hash Format:</strong> Hexadecimal (a-f, 0-9)</p>
        </div>
        
        <div class="info">
            <h3>⚙️ How to Use</h3>
            <ol>
                <li>Copy the hash above</li>
                <li>Use it in one of the URL formats shown</li>
                <li>Each IP address can access the protected page up to 3 times</li>
                <li>After 3 attempts, the IP will be blocked for 30 minutes</li>
                <li>Each session with IP will be tracked</li>
            </ol>
        </div>
        
        <hr>
        <p><a href="protected-page.php">← Back to Protected Page (without hash = access denied)</a></p>
        <p><a href="generate-hash.php">🔄 Generate New Hash</a></p>
    </div>
</body>
</html>
