<?php
/**
 * Hash-based Access Control with IP Rate Limiting
 * This file must be included at the top of protected pages
 */

error_reporting(0);
session_start();

// Configuration
define('HASH_LENGTH', 512);
define('MAX_ATTEMPTS', 3);
define('BLOCK_DURATION', 30 * 60); // 30 minutes in seconds
define('IP_TRACKING_DIR', __DIR__ . '/../ip-tracking/');

// Include IP detection
require_once __DIR__ . '/../detect-ip.php';

// Create storage directory if it doesn't exist
if (!file_exists(IP_TRACKING_DIR)) {
    mkdir(IP_TRACKING_DIR, 0755, true);
}

/**
 * Get hash from URL
 */
function getHashFromURL() {
    $requestUri = $_SERVER['REQUEST_URI'];
    
    // Remove query string if exists
    $path = parse_url($requestUri, PHP_URL_PATH);
    
    // Extract hash from path (assumes format: /hash512/ or /hash512)
    $pathParts = explode('/', trim($path, '/'));
    
    foreach ($pathParts as $part) {
        if (strlen($part) === HASH_LENGTH && preg_match('/^[a-f0-9]+$/', $part)) {
            return $part;
        }
    }
    
    // Also check if hash is passed as GET parameter
    if (isset($_GET['hash']) && strlen($_GET['hash']) === HASH_LENGTH) {
        return $_GET['hash'];
    }
    
    return false;
}

/**
 * Get IP tracking file path
 */
function getIPTrackingFile($ip) {
    $safeIP = preg_replace('/[^a-zA-Z0-9\-\.]/', '-', $ip);
    return IP_TRACKING_DIR . 'ip_' . $safeIP . '.json';
}

/**
 * Check if IP is blocked
 */
function checkIPStatus($ip) {
    $trackingFile = getIPTrackingFile($ip);
    
    if (!file_exists($trackingFile)) {
        return [
            'blocked' => false,
            'until' => null,
            'attempts' => 0,
            'access_times' => []
        ];
    }
    
    $data = json_decode(file_get_contents($trackingFile), true);
    if (!$data) {
        return [
            'blocked' => false,
            'until' => null,
            'attempts' => 0,
            'access_times' => []
        ];
    }
    
    $currentTime = time();
    $accessTimes = isset($data['access_times']) ? $data['access_times'] : [];
    
    // Check if IP is blocked
    if (isset($data['blocked_until']) && $data['blocked_until'] > $currentTime) {
        return [
            'blocked' => true,
            'until' => $data['blocked_until'],
            'attempts' => count($accessTimes),
            'access_times' => $accessTimes
        ];
    }
    
    // If block expired, reset
    if (isset($data['blocked_until']) && $data['blocked_until'] <= $currentTime) {
        unlink($trackingFile);
        return [
            'blocked' => false,
            'until' => null,
            'attempts' => 0,
            'access_times' => []
        ];
    }
    
    // Clean old access times
    $validAccessTimes = [];
    foreach ($accessTimes as $accessTime) {
        if ($accessTime > ($currentTime - BLOCK_DURATION)) {
            $validAccessTimes[] = $accessTime;
        }
    }
    
    // Update if we cleaned old entries
    if (count($validAccessTimes) !== count($accessTimes)) {
        $data['access_times'] = $validAccessTimes;
        file_put_contents($trackingFile, json_encode($data));
    }
    
    return [
        'blocked' => false,
        'until' => null,
        'attempts' => count($validAccessTimes),
        'access_times' => $validAccessTimes
    ];
}

/**
 * Record IP access
 */
function recordIPAccess($ip, $hash) {
    $trackingFile = getIPTrackingFile($ip);
    $currentTime = time();
    
    $data = [];
    if (file_exists($trackingFile)) {
        $data = json_decode(file_get_contents($trackingFile), true) ?: [];
    }
    
    if (!isset($data['access_times'])) {
        $data['access_times'] = [];
    }
    
    // Add current access time
    $data['access_times'][] = $currentTime;
    
    // Clean old access times
    $validAccessTimes = [];
    foreach ($data['access_times'] as $accessTime) {
        if ($accessTime > ($currentTime - BLOCK_DURATION)) {
            $validAccessTimes[] = $accessTime;
        }
    }
    $data['access_times'] = $validAccessTimes;
    
    $attempts = count($validAccessTimes);
    
    // Check if should be blocked
    if ($attempts >= MAX_ATTEMPTS) {
        $data['blocked_until'] = $currentTime + BLOCK_DURATION;
        $data['blocked_at'] = $currentTime;
    }
    
    $data['last_access'] = $currentTime;
    $data['ip'] = $ip;
    $data['hash_used'] = $hash;
    
    file_put_contents($trackingFile, json_encode($data));
    
    return [
        'blocked' => ($attempts >= MAX_ATTEMPTS),
        'until' => $data['blocked_until'] ?? null,
        'attempts' => $attempts
    ];
}

/**
 * Create session with IP
 */
function createSessionWithIP($ip, $hash) {
    $sessionId = session_id();
    if (empty($sessionId)) {
        session_start();
        $sessionId = session_id();
    }
    
    // Store in session
    $_SESSION['protected_hash'] = $hash;
    $_SESSION['protected_ip'] = $ip;
    $_SESSION['protected_session'] = true;
    $_SESSION['session_created'] = time();
    
    // Also store in file for tracking
    $sessionFile = IP_TRACKING_DIR . 'session_' . $sessionId . '.json';
    $sessionData = [
        'session_id' => $sessionId,
        'ip' => $ip,
        'hash' => $hash,
        'created_at' => time(),
        'last_access' => time(),
        'access_count' => isset($_SESSION['access_count']) ? ($_SESSION['access_count'] + 1) : 1
    ];
    
    if (file_exists($sessionFile)) {
        $existing = json_decode(file_get_contents($sessionFile), true);
        if ($existing) {
            $sessionData['access_count'] = ($existing['access_count'] ?? 0) + 1;
            $sessionData['created_at'] = $existing['created_at'];
        }
    }
    
    $_SESSION['access_count'] = $sessionData['access_count'];
    
    file_put_contents($sessionFile, json_encode($sessionData));
}

// Main protection logic
$ip = detect_ip();
$hash = getHashFromURL();

// Check if hash is provided
if (!$hash || strlen($hash) !== HASH_LENGTH) {
    http_response_code(403);
    die("Access Denied: Valid 512-character hash required in URL. Format: domain.com/{hash}/");
}

// Check IP status
$ipStatus = checkIPStatus($ip);

// If IP is blocked
if ($ipStatus['blocked']) {
    $blockedUntil = date('Y-m-d H:i:s', $ipStatus['until']);
    $remainingMinutes = round(($ipStatus['until'] - time()) / 60);
    
    http_response_code(429);
    die("Access Denied: IP address blocked. Maximum attempts (3) exceeded. Blocked until: {$blockedUntil} ({$remainingMinutes} minutes remaining)");
}

// Record access
$accessResult = recordIPAccess($ip, $hash);

// If this access triggered a block
if ($accessResult['blocked']) {
    $blockedUntil = date('Y-m-d H:i:s', $accessResult['until']);
    $remainingMinutes = round(($accessResult['until'] - time()) / 60);
    
    http_response_code(429);
    die("Access Denied: IP address blocked. Maximum attempts (3) exceeded. Blocked until: {$blockedUntil} ({$remainingMinutes} minutes remaining)");
}

// Create/update session with IP
createSessionWithIP($ip, $hash);

// Store attempt count for reference
$remainingAttempts = MAX_ATTEMPTS - ($ipStatus['attempts'] + 1);
$_SESSION['remaining_attempts'] = max(0, $remainingAttempts);

?>
