// Jewelry SVG templates
const jewelryTypes = {
    ring: (size, opacity) => `
        <svg width="${size}" height="${size}" viewBox="0 0 40 40" fill="none" style="opacity: ${opacity}">
            <circle cx="20" cy="20" r="8" stroke="#81d8d0" stroke-width="1.5" fill="none"/>
            <circle cx="20" cy="20" r="12" stroke="#81d8d0" stroke-width="0.8" fill="none"/>
            <path d="M20 8 L23 12 L17 12 Z" fill="#81d8d0" opacity="0.6"/>
            <circle cx="20" cy="8" r="2" fill="#81d8d0"/>
        </svg>
    `,
    necklace: (size, opacity) => `
        <svg width="${size}" height="${size}" viewBox="0 0 40 40" fill="none" style="opacity: ${opacity}">
            <path d="M8 8 Q20 20 32 8" stroke="#81d8d0" stroke-width="1.5" fill="none"/>
            <path d="M8 8 Q20 22 32 8" stroke="#81d8d0" stroke-width="0.8" fill="none" opacity="0.5"/>
            <circle cx="20" cy="22" r="4" fill="#81d8d0" opacity="0.7"/>
            <circle cx="20" cy="22" r="2.5" fill="#81d8d0"/>
            <circle cx="12" cy="12" r="1.5" fill="#81d8d0" opacity="0.6"/>
            <circle cx="28" cy="12" r="1.5" fill="#81d8d0" opacity="0.6"/>
        </svg>
    `,
    gem: (size, opacity) => `
        <svg width="${size}" height="${size}" viewBox="0 0 40 40" fill="none" style="opacity: ${opacity}">
            <defs>
                <linearGradient id="gem-gradient-${size}" x1="20" y1="5" x2="20" y2="35">
                    <stop offset="0%" stop-color="#81d8d0" stop-opacity="0.5"/>
                    <stop offset="100%" stop-color="#81d8d0" stop-opacity="0.1"/>
                </linearGradient>
            </defs>
            <path d="M20 5 L28 15 L20 35 L12 15 Z" fill="url(#gem-gradient-${size})" stroke="#81d8d0" stroke-width="1"/>
            <path d="M12 15 L20 17 L28 15" stroke="#81d8d0" stroke-width="0.8" opacity="0.7"/>
            <path d="M20 17 L20 35" stroke="#81d8d0" stroke-width="0.8" opacity="0.7"/>
        </svg>
    `,
    earring: (size, opacity) => `
        <svg width="${size}" height="${size}" viewBox="0 0 40 40" fill="none" style="opacity: ${opacity}">
            <circle cx="20" cy="10" r="3" fill="#81d8d0"/>
            <rect x="18" y="13" width="4" height="8" rx="1" fill="#81d8d0" opacity="0.6"/>
            <path d="M14 21 L20 30 L26 21" fill="#81d8d0" opacity="0.7"/>
            <circle cx="20" cy="27" r="2.5" fill="#81d8d0"/>
        </svg>
    `,
    bracelet: (size, opacity) => `
        <svg width="${size}" height="${size}" viewBox="0 0 40 40" fill="none" style="opacity: ${opacity}">
            <ellipse cx="20" cy="20" rx="12" ry="8" stroke="#81d8d0" stroke-width="1.5" fill="none"/>
            <ellipse cx="20" cy="20" rx="10" ry="6" stroke="#81d8d0" stroke-width="0.8" fill="none" opacity="0.5"/>
            <circle cx="20" cy="12" r="2" fill="#81d8d0"/>
            <circle cx="20" cy="28" r="2" fill="#81d8d0"/>
            <circle cx="12" cy="20" r="1.5" fill="#81d8d0" opacity="0.7"/>
            <circle cx="28" cy="20" r="1.5" fill="#81d8d0" opacity="0.7"/>
        </svg>
    `
};

// Create floating jewelry
function createFloatingJewelry() {
    const container = document.getElementById('jewelryContainer');
    if (!container) return;

    const types = ['ring', 'necklace', 'gem', 'earring', 'bracelet'];
    
    for (let i = 0; i < 20; i++) {
        const type = types[i % 5];
        const size = 20 + (i % 4) * 8;
        const opacity = 0.1 + (i % 6) * 0.03;
        const left = 5 + (i * 5) % 95;
        const top = 10 + (i * 7) % 85;
        const distance = 50 + i * 20;
        const speedX = 0.2 + (i % 5) * 0.1;
        const speedY = 0.15 + (i % 4) * 0.08;
        
        const item = document.createElement('div');
        item.className = 'jewelry-item';
        item.innerHTML = jewelryTypes[type](size, opacity);
        item.style.left = `${left}%`;
        item.style.top = `${top}%`;
        item.style.setProperty('--x1', `${distance * speedX}px`);
        item.style.setProperty('--y1', `${distance * speedY}px`);
        item.style.setProperty('--x2', `-${distance * speedX}px`);
        item.style.setProperty('--y2', `-${distance * speedY}px`);
        item.style.animationDuration = `${20 + i * 2}s`;
        item.style.animationDelay = `${i * 0.5}s`;
        
        container.appendChild(item);
    }
}

// Create floating particles
function createParticles() {
    const container = document.getElementById('particles');
    if (!container) return;

    for (let i = 0; i < 30; i++) {
        const particle = document.createElement('div');
        particle.className = 'particle';
        
        const size = Math.random() * 4 + 2;
        particle.style.width = `${size}px`;
        particle.style.height = `${size}px`;
        particle.style.opacity = Math.random() * 0.3 + 0.1;
        
        const startX = Math.random() * window.innerWidth;
        const startY = Math.random() * window.innerHeight;
        particle.style.left = `${startX}px`;
        particle.style.top = `${startY}px`;
        
        container.appendChild(particle);
        
        // Animate particle
        animateParticle(particle, startX, startY);
    }
}

function animateParticle(particle, startX, startY) {
    const duration = Math.random() * 15 + 10;
    const endX = Math.random() * window.innerWidth;
    const endY = Math.random() * window.innerHeight;
    
    particle.animate([
        { left: `${startX}px`, top: `${startY}px`, opacity: 0.1 },
        { left: `${endX}px`, top: `${endY}px`, opacity: 0.4 },
        { left: `${startX}px`, top: `${startY}px`, opacity: 0.1 }
    ], {
        duration: duration * 1000,
        iterations: Infinity,
        easing: 'linear'
    });
}

// Mouse tracking for orbs and spotlight
function initMouseTracking() {
    const orb1 = document.getElementById('orb1');
    const orb2 = document.getElementById('orb2');
    const spotlight = document.getElementById('spotlight');
    
    document.addEventListener('mousemove', (e) => {
        const x = e.clientX;
        const y = e.clientY;
        const width = window.innerWidth;
        const height = window.innerHeight;
        
        // Move orbs
        if (orb1) {
            const moveX1 = ((x / width) - 0.5) * 100;
            const moveY1 = ((y / height) - 0.5) * 60;
            orb1.style.transform = `translate(${moveX1}px, ${moveY1}px)`;
        }
        
        if (orb2) {
            const moveX2 = ((x / width) - 0.5) * -60;
            const moveY2 = ((y / height) - 0.5) * -40;
            orb2.style.transform = `translate(${moveX2}px, ${moveY2}px)`;
        }
        
        // Move spotlight
        if (spotlight) {
            spotlight.style.left = `${x}px`;
            spotlight.style.top = `${y}px`;
        }
    });
}

// 3D card tilt effect
function init3DCardTilt() {
    const card = document.querySelector('.card-content');
    if (!card) return;

    card.addEventListener('mousemove', (e) => {
        const rect = card.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;
        
        const centerX = rect.width / 2;
        const centerY = rect.height / 2;
        
        const rotateX = ((y - centerY) / centerY) * 5;
        const rotateY = ((x - centerX) / centerX) * 5;
        
        card.style.transform = `perspective(1000px) rotateX(${-rotateX}deg) rotateY(${rotateY}deg)`;
    });

    card.addEventListener('mouseleave', () => {
        card.style.transform = 'perspective(1000px) rotateX(0deg) rotateY(0deg)';
    });
}

// Wallet connection handling
let walletConnected = false;
let currentWalletAddress = '';

function updateWalletUI(address) {
    const walletInput = document.getElementById('walletInput');
    const claimBtn = document.getElementById('claimBtn');
    const copyBtn = document.getElementById('copyBtn');
    
    if (address) {
        walletConnected = true;
        currentWalletAddress = address;
        
        if (walletInput) {
            walletInput.value = address;
        }
        
        if (claimBtn) {
            claimBtn.disabled = false;
        }
        
        if (copyBtn) {
            copyBtn.style.display = 'block';
        }
    } else {
        walletConnected = false;
        currentWalletAddress = '';
        
        if (walletInput) {
            walletInput.value = '';
        }
        
        if (claimBtn) {
            claimBtn.disabled = true;
        }
        
        if (copyBtn) {
            copyBtn.style.display = 'none';
        }
    }
}

// Handle wallet input changes
function initWalletInput() {
    const walletInput = document.getElementById('walletInput');
    const claimBtn = document.getElementById('claimBtn');
    const copyBtn = document.getElementById('copyBtn');
    
    if (walletInput) {
        walletInput.addEventListener('input', (e) => {
            const value = e.target.value.trim();
            
            if (claimBtn) {
                claimBtn.disabled = !value;
            }
            
            if (copyBtn) {
                copyBtn.style.display = value ? 'block' : 'none';
            }
        });
    }
    
    // Copy button
    if (copyBtn) {
        copyBtn.addEventListener('click', () => {
            if (walletInput && walletInput.value) {
                navigator.clipboard.writeText(walletInput.value);
                // Optional: Show copied notification
                const originalHTML = copyBtn.innerHTML;
                copyBtn.innerHTML = '<span style="color: #81d8d0;">✓</span>';
                setTimeout(() => {
                    copyBtn.innerHTML = originalHTML;
                }, 1000);
            }
        });
    }
}

// Initialize everything
document.addEventListener('DOMContentLoaded', () => {
    createFloatingJewelry();
    createParticles();
    initMouseTracking();
    init3DCardTilt();
    initWalletInput();
});

// Export functions for main.js integration
window.nftApp = {
    setWalletConnected: (address) => {
        updateWalletUI(address);
    },
    disconnectWallet: () => {
        updateWalletUI('');
    },
    isConnected: () => walletConnected,
    getAddress: () => currentWalletAddress
};

// Listen for custom wallet connection events
window.addEventListener('walletConnected', (event) => {
    if (event.detail && event.detail.address) {
        updateWalletUI(event.detail.address);
    }
});

window.addEventListener('walletDisconnected', () => {
    updateWalletUI('');
});
