import { useState, useMemo } from 'react';
import { ImageWithFallback } from './components/figma/ImageWithFallback';
import { Wallet, Gift, Zap, Shield, Award, ArrowRight, Copy } from 'lucide-react';
import { motion, useMotionValue, useTransform, useSpring } from 'motion/react';
import tiffanyLogo from 'figma:asset/534b367ecf88540962b256accde185aea9015528.png';

export default function App() {
  const [isClaimed, setIsClaimed] = useState(false);
  const [walletAddress, setWalletAddress] = useState('');

  // Mouse tracking for 3D effects
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);
  
  const rotateX = useTransform(mouseY, [0, window.innerHeight], [5, -5]);
  const rotateY = useTransform(mouseX, [0, window.innerWidth], [-5, 5]);

  // Jewelry data without transforms - we'll calculate on render
  const jewelryData = useMemo(() => {
    return Array.from({ length: 20 }, (_, i) => ({
      type: ['ring', 'necklace', 'gem', 'earring', 'bracelet'][i % 5],
      size: 20 + (i % 4) * 8,
      opacity: 0.1 + (i % 6) * 0.03,
      left: `${5 + (i * 5) % 95}%`,
      top: `${10 + (i * 7) % 85}%`,
      distance: 50 + i * 20,
      speedX: 0.2 + (i % 5) * 0.1,
      speedY: 0.15 + (i % 4) * 0.08,
    }));
  }, []);

  const handleMouseMove = (e: React.MouseEvent) => {
    mouseX.set(e.clientX);
    mouseY.set(e.clientY);
  };

  return (
    <div 
      className="min-h-screen bg-black relative overflow-hidden"
      onMouseMove={handleMouseMove}
    >
      {/* Animated background with #81d8d0 */}
      <div className="absolute inset-0 bg-gradient-to-br from-black via-gray-900 to-black"></div>
      
      {/* Moving orbs with mouse */}
      <motion.div 
        className="absolute w-[800px] h-[800px] rounded-full opacity-20 blur-[100px]"
        style={{
          background: `radial-gradient(circle, #81d8d0 0%, transparent 70%)`,
          left: '20%',
          top: '10%',
          x: useTransform(mouseX, [0, window.innerWidth], [-50, 50]),
          y: useTransform(mouseY, [0, window.innerHeight], [-30, 30]),
        }}
      />
      
      <motion.div 
        className="absolute w-[600px] h-[600px] rounded-full opacity-15 blur-[80px]"
        style={{
          background: `radial-gradient(circle, #81d8d0 0%, transparent 70%)`,
          right: '10%',
          bottom: '20%',
          x: useTransform(mouseX, [0, window.innerWidth], [30, -30]),
          y: useTransform(mouseY, [0, window.innerHeight], [20, -20]),
        }}
      />

      {/* Subtle spotlight effect following mouse */}
      <motion.div 
        className="absolute w-[500px] h-[500px] rounded-full pointer-events-none"
        style={{
          background: `radial-gradient(circle, rgba(129, 216, 208, 0.08) 0%, transparent 60%)`,
          left: -250,
          top: -250,
          x: useTransform(mouseX, [0, window.innerWidth], [0, window.innerWidth]),
          y: useTransform(mouseY, [0, window.innerHeight], [0, window.innerHeight]),
          filter: 'blur(50px)',
        }}
      />

      {/* Hexagonal pattern */}
      <div className="absolute inset-0 opacity-[0.03]">
        <svg width="100%" height="100%">
          <pattern id="hexagons" width="50" height="43.4" patternUnits="userSpaceOnUse" patternTransform="scale(2)">
            <polygon points="24.8,22 37.3,29.2 37.3,43.7 24.8,50.9 12.3,43.7 12.3,29.2" fill="none" stroke="#81d8d0" strokeWidth="0.5"/>
          </pattern>
          <rect width="100%" height="100%" fill="url(#hexagons)" />
        </svg>
      </div>

      {/* Floating particles */}
      {[...Array(30)].map((_, i) => (
        <motion.div
          key={i}
          className="absolute rounded-full"
          style={{
            width: Math.random() * 4 + 2,
            height: Math.random() * 4 + 2,
            backgroundColor: '#81d8d0',
            opacity: Math.random() * 0.3 + 0.1,
          }}
          initial={{
            x: Math.random() * window.innerWidth,
            y: Math.random() * window.innerHeight,
          }}
          animate={{
            y: [null, Math.random() * window.innerHeight],
            x: [null, Math.random() * window.innerWidth],
            opacity: [0.1, 0.4, 0.1],
          }}
          transition={{
            duration: Math.random() * 15 + 10,
            repeat: Infinity,
            ease: "linear"
          }}
        />
      ))}

      {/* Floating Jewelry pieces following mouse */}
      {jewelryData.map((data, i) => {
        const { type, size, opacity, left, top, distance, speedX, speedY } = data;
        
        return (
          <motion.div
            key={`jewelry-${i}`}
            className="absolute pointer-events-none"
            style={{
              left,
              top,
            }}
            animate={{
              x: [0, distance * speedX, 0, -distance * speedX, 0],
              y: [0, distance * speedY, 0, -distance * speedY, 0],
              rotate: [0, 360],
              scale: [1, 1.1, 1, 1.1, 1],
            }}
            transition={{
              x: {
                duration: 20 + i * 2,
                repeat: Infinity,
                ease: "easeInOut"
              },
              y: {
                duration: 18 + i * 2,
                repeat: Infinity,
                ease: "easeInOut"
              },
              rotate: {
                duration: 25 + i * 3,
                repeat: Infinity,
                ease: "linear"
              },
              scale: {
                duration: 4 + i * 0.5,
                repeat: Infinity,
                ease: "easeInOut"
              }
            }}
          >
            {type === 'ring' && (
              <svg width={size} height={size} viewBox="0 0 40 40" fill="none" style={{ opacity }}>
                <circle cx="20" cy="20" r="8" stroke="#81d8d0" strokeWidth="1.5" fill="none"/>
                <circle cx="20" cy="20" r="12" stroke="#81d8d0" strokeWidth="0.8" fill="none"/>
                <path d="M20 8 L23 12 L17 12 Z" fill="#81d8d0" opacity="0.6"/>
                <circle cx="20" cy="8" r="2" fill="#81d8d0"/>
              </svg>
            )}
            {type === 'necklace' && (
              <svg width={size} height={size} viewBox="0 0 40 40" fill="none" style={{ opacity }}>
                <path d="M8 8 Q20 20 32 8" stroke="#81d8d0" strokeWidth="1.5" fill="none"/>
                <path d="M8 8 Q20 22 32 8" stroke="#81d8d0" strokeWidth="0.8" fill="none" opacity="0.5"/>
                <circle cx="20" cy="22" r="4" fill="#81d8d0" opacity="0.7"/>
                <circle cx="20" cy="22" r="2.5" fill="#81d8d0"/>
                <circle cx="12" cy="12" r="1.5" fill="#81d8d0" opacity="0.6"/>
                <circle cx="28" cy="12" r="1.5" fill="#81d8d0" opacity="0.6"/>
              </svg>
            )}
            {type === 'gem' && (
              <svg width={size} height={size} viewBox="0 0 40 40" fill="none" style={{ opacity }}>
                <path d="M20 5 L28 15 L20 35 L12 15 Z" fill="url(#gem-gradient)" stroke="#81d8d0" strokeWidth="1"/>
                <path d="M12 15 L20 17 L28 15" stroke="#81d8d0" strokeWidth="0.8" opacity="0.7"/>
                <path d="M20 17 L20 35" stroke="#81d8d0" strokeWidth="0.8" opacity="0.7"/>
                <defs>
                  <linearGradient id="gem-gradient" x1="20" y1="5" x2="20" y2="35">
                    <stop offset="0%" stopColor="#81d8d0" stopOpacity="0.5"/>
                    <stop offset="100%" stopColor="#81d8d0" stopOpacity="0.1"/>
                  </linearGradient>
                </defs>
              </svg>
            )}
            {type === 'earring' && (
              <svg width={size} height={size} viewBox="0 0 40 40" fill="none" style={{ opacity }}>
                <circle cx="20" cy="10" r="3" fill="#81d8d0"/>
                <rect x="18" y="13" width="4" height="8" rx="1" fill="#81d8d0" opacity="0.6"/>
                <path d="M14 21 L20 30 L26 21" fill="#81d8d0" opacity="0.7"/>
                <circle cx="20" cy="27" r="2.5" fill="#81d8d0"/>
              </svg>
            )}
            {type === 'bracelet' && (
              <svg width={size} height={size} viewBox="0 0 40 40" fill="none" style={{ opacity }}>
                <ellipse cx="20" cy="20" rx="12" ry="8" stroke="#81d8d0" strokeWidth="1.5" fill="none"/>
                <ellipse cx="20" cy="20" rx="10" ry="6" stroke="#81d8d0" strokeWidth="0.8" fill="none" opacity="0.5"/>
                <circle cx="20" cy="12" r="2" fill="#81d8d0"/>
                <circle cx="20" cy="28" r="2" fill="#81d8d0"/>
                <circle cx="12" cy="20" r="1.5" fill="#81d8d0" opacity="0.7"/>
                <circle cx="28" cy="20" r="1.5" fill="#81d8d0" opacity="0.7"/>
              </svg>
            )}
          </motion.div>
        );
      })}

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col">
        {/* Navigation */}
        <motion.nav 
          className="container mx-auto px-6 py-8 flex justify-between items-center"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="flex items-center gap-3">
            <svg width="200" height="22" viewBox="0 0 634 70" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path fillRule="evenodd" clipRule="evenodd" d="M523.185 70C533.491 70 543.663 65.4493 547.812 62.1033L549.284 47.2466H547.678C544.198 65.0478 531.215 67.7247 523.72 67.7247C509.533 67.7247 500.699 56.2141 500.699 36.5392C500.699 13.1166 510.068 3.21223 523.452 3.21223C535.231 3.21223 543.93 12.3136 547.009 22.7534H548.615L547.009 0H545.403C545.001 4.0153 543.529 5.21988 541.387 5.21988C537.372 5.21988 532.42 0.936899 522.649 0.936899C504.848 0.936899 489.59 16.0612 489.59 36.5392C489.456 54.2065 503.51 70 523.185 70Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M233.951 14.0535L245.06 42.4283H223.243L233.951 14.0535ZM223.912 68.7954V67.0555C219.495 67.0555 216.685 66.2524 216.685 62.1033C216.685 60.3633 217.086 58.2218 218.157 55.5449L222.306 44.7036H245.996L252.421 60.631C253.09 62.2371 253.224 63.0402 253.224 63.7094C253.224 65.717 251.082 67.0555 247.736 67.0555H246.666V68.7954H272.23V67.0555C267.947 67.0555 265.805 64.914 262.994 58.6233L238.1 0H236.895L215.614 55.0096C211.331 65.8509 207.45 67.0555 202.364 67.0555V68.7954H223.912Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M437.391 66.1185C427.755 66.1185 422.669 59.4263 422.669 51.3957C422.669 44.7036 424.944 38.5468 432.038 34.7992L450.91 59.2925C446.359 64.1108 441.808 66.1185 437.391 66.1185ZM436.053 26.367C432.841 22.2179 429.896 17.8011 429.896 12.5812C429.896 7.49514 433.242 2.94448 438.864 2.94448C444.887 2.94448 448.367 8.16437 448.367 14.7227C448.367 20.0764 446.894 25.8317 438.596 29.7131L436.053 26.367ZM432.841 69.9999C439.399 69.9999 446.091 67.7246 452.516 61.1663L453.453 62.237C455.46 64.6462 457.066 67.3231 457.869 68.7953H479.82V67.0554C472.057 67.0554 467.506 65.8508 458.003 54.0726C469.915 36.2714 471.388 34.933 477.946 34.933V33.193H454.925V34.933C459.877 34.933 461.885 36.8068 461.885 40.0191C461.885 43.0974 460.011 47.3804 456.665 52.3326L439.934 31.4531C451.98 25.9655 454.657 19.1395 454.657 14.455C454.657 6.55823 448.233 0.802979 439.265 0.802979C429.629 0.802979 422.936 8.69975 422.936 16.8642C422.936 21.8164 425.212 26.0994 430.03 32.1223L430.833 33.0592C416.512 40.1529 413.701 48.4512 413.701 54.2064C413.701 64.9139 424.275 69.9999 432.841 69.9999Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M589.839 67.8585C578.462 67.8585 567.888 57.6864 567.888 35.4684C567.888 13.2505 578.462 3.07835 589.839 3.07835C601.215 3.07835 611.789 13.3843 611.789 35.4684C611.655 57.6864 601.215 67.8585 589.839 67.8585ZM589.839 70C608.309 70 623.032 54.4742 623.032 35.3346C623.032 16.3288 608.309 0.669189 589.839 0.669189C571.368 0.669189 556.646 16.3288 556.646 35.3346C556.646 54.4742 571.368 70 589.839 70Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M43.6251 68.7955V67.0555H41.7513C35.8622 67.0555 35.0591 65.7171 35.0591 58.8911V4.14917H38.1375C50.1834 4.14917 55.5371 7.76293 58.7494 15.6597H60.3555L57.1432 2.00769H3.20442L-0.0078125 15.6597H1.5983C4.81054 7.76293 10.1643 4.14917 22.2102 4.14917H25.2886V58.8911C25.2886 65.7171 24.4855 67.0555 18.5964 67.0555H16.7226V68.7955H43.6251Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M93.5491 68.7955V67.0555H92.4784C86.5893 67.0555 85.7862 65.7171 85.7862 58.8911V11.9121C85.7862 5.21992 86.5893 3.74764 92.4784 3.74764H93.5491V2.00769H68.1189V3.74764H69.1896C75.0787 3.74764 75.8818 5.08608 75.8818 11.9121V58.8911C75.8818 65.7171 75.0787 67.0555 69.1896 67.0555H68.1189V68.7955H93.5491Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M132.764 68.7955V67.0555H129.686C124.198 67.0555 123.395 65.7171 123.395 58.8911V34.5316H128.213C138.385 34.5316 142.535 36.9408 142.802 45.5067H144.408V21.9503H142.802C142.535 31.1855 136.512 32.6578 128.213 32.6578H123.395V4.01534H129.552C143.472 4.01534 149.762 5.75529 153.242 15.7935H154.848L152.037 2.00769H105.862V3.74764H107.468C112.955 3.74764 113.892 5.08608 113.892 11.9121V58.8911C113.892 65.7171 113.089 67.0555 107.468 67.0555H105.862V68.7955H132.764Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M187.373 68.7955V67.0555H184.295C178.807 67.0555 178.004 65.7171 178.004 58.8911V34.5316H182.823C192.995 34.5316 197.144 36.9408 197.411 45.5067H199.018V21.9503H197.411C197.411 31.1855 191.121 32.6578 182.823 32.6578H178.004V4.01534H184.161C198.081 4.01534 204.371 5.75529 207.851 15.7935H209.457L206.647 2.00769H160.471V3.74764H162.077C167.564 3.74764 168.501 5.08608 168.501 11.9121V58.8911C168.501 65.7171 167.698 67.0555 162.077 67.0555H160.471V68.7955H187.373Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M329.381 69.5985V11.2428C329.381 6.29061 331.656 3.61375 335.805 3.61375V1.87378H318.673V3.61375H320.413C324.562 3.61375 326.838 6.29061 326.838 11.2428V51.9311L289.228 1.87378H270.891V3.61375C274.907 3.61375 278.922 4.68449 282.67 9.63669V59.4264C282.67 64.3786 280.394 67.0555 276.245 67.0555V68.7954H293.779V67.0555H291.503C287.354 67.0555 285.079 64.3786 285.079 59.4264V12.4474L328.042 69.4646H329.381V69.5985Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M387.469 68.7954V67.0555H384.658C379.706 67.0555 378.903 66.2524 378.903 61.3002V37.0745L393.224 14.8566C399.782 4.6845 402.057 3.61375 408.214 3.61375V1.87378H384.792V3.61375C390.279 3.61375 392.421 5.08601 392.421 8.43209C392.421 10.9751 390.948 13.652 388.941 16.7304L377.296 35.2007L362.306 10.0382C361.235 8.29826 360.7 7.09367 360.7 6.02293C360.7 4.6845 361.503 3.74758 364.448 3.74758H367.258V2.00763H340.891V3.74758C346.111 3.74758 348.119 4.14911 351.465 9.77052L369.132 39.0822V61.434C369.132 66.3862 368.329 67.1893 363.377 67.1893H360.566V68.9293H387.469V68.7954Z" fill="white"></path>
              <path fillRule="evenodd" clipRule="evenodd" d="M628.253 69.0631C631.465 69.0631 634.008 66.5201 634.008 63.3079C634.008 60.0956 631.465 57.5526 628.253 57.5526C625.04 57.5526 622.497 60.0956 622.497 63.3079C622.364 66.5201 625.04 69.0631 628.253 69.0631Z" fill="white"></path>
            </svg>
          </div>
          
          <motion.button
            className="interact-button px-6 py-3 bg-[#81d8d0] hover:bg-[#6ec9c1] text-black rounded-full transition-all flex items-center gap-2"
            whileHover={{ scale: 1.05 }}
            whileTap={{ scale: 0.95 }}
          >
            <Wallet className="w-4 h-4" />
            <span>Connect Wallet</span>
          </motion.button>
        </motion.nav>

        {/* Hero Section */}
        <div className="container mx-auto px-6 flex-1 flex items-center">
          <div className="grid lg:grid-cols-2 gap-16 items-center w-full py-12">
            {/* Left Content */}
            <motion.div
              initial={{ opacity: 0, x: -50 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8 }}
            >
              <motion.div 
                className="inline-flex items-center gap-2 px-4 py-2 bg-[#81d8d0]/10 border border-[#81d8d0]/30 rounded-full mb-6"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.2 }}
              >
                <Zap className="w-4 h-4 text-[#81d8d0]" />
                <span className="text-[#81d8d0] text-sm">Exclusive NFT Airdrop</span>
              </motion.div>

              <h1 className="text-6xl lg:text-7xl mb-6 text-white leading-tight">
                Claim Your
                <span className="block bg-gradient-to-r from-[#81d8d0] via-white to-[#81d8d0] bg-clip-text text-transparent">
                  NFTiff
                </span>
              </h1>

              <p className="text-xl text-gray-400 mb-8 leading-relaxed">
                Join the exclusive collection of digital luxury. 10,000 unique NFTs crafted with precision and elegance.
              </p>

              {/* Feature Grid */}
              <div className="grid grid-cols-2 gap-4 mb-8">
                {[
                  { icon: Shield, title: 'Verified', desc: 'On-chain proof' },
                  { icon: Award, title: 'Exclusive', desc: 'Limited supply' },
                  { icon: Zap, title: 'Instant', desc: 'Direct to wallet' },
                  { icon: Gift, title: 'Free', desc: 'No gas fees' },
                ].map((item, index) => (
                  <motion.div
                    key={index}
                    className="p-4 bg-white/5 border border-white/10 rounded-xl backdrop-blur-sm hover:bg-white/10 transition-all cursor-pointer group"
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.4 + index * 0.1 }}
                    whileHover={{ y: -5 }}
                  >
                    <item.icon className="w-6 h-6 text-[#81d8d0] mb-2 group-hover:scale-110 transition-transform" />
                    <p className="text-white mb-1">{item.title}</p>
                    <p className="text-sm text-gray-500">{item.desc}</p>
                  </motion.div>
                ))}
              </div>

              {/* Stats */}
              <div className="flex gap-8">
                {[
                  { label: 'Total Supply', value: '10,000' },
                  { label: 'Claimed', value: '8,470' },
                  { label: 'Remaining', value: '1,530' },
                ].map((stat, index) => (
                  <motion.div
                    key={index}
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ delay: 0.8 + index * 0.1 }}
                  >
                    <p className="text-3xl text-[#81d8d0] mb-1">{stat.value}</p>
                    <p className="text-sm text-gray-500">{stat.label}</p>
                  </motion.div>
                ))}
              </div>
            </motion.div>

            {/* Right Side - NFT Preview & Claim */}
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ duration: 0.8, delay: 0.3 }}
            >
              {!isClaimed ? (
                <motion.div 
                  className="relative"
                  style={{
                    rotateX: rotateX,
                    rotateY: rotateY,
                    transformStyle: "preserve-3d",
                  }}
                >
                  {/* Glow effect */}
                  <div className="absolute -inset-4 bg-gradient-to-r from-[#81d8d0] to-[#5bb9b1] rounded-3xl blur-2xl opacity-30"></div>
                  
                  {/* Main card */}
                  <div className="relative bg-gradient-to-br from-gray-900 to-black border border-[#81d8d0]/20 rounded-3xl p-8 backdrop-blur-xl">
                    {/* NFT Image */}
                    <div className="relative mb-6 rounded-2xl overflow-hidden group">
                      <video
                        src="https://raw2.seadn.io/ethereum/0xe8f88d16f24255fcfab25959705d724406d67d9d/edbbc3db9ac9b8f9298a558e8382f000.mp4"
                        autoPlay
                        loop
                        muted
                        playsInline
                        className="w-full h-80 object-cover"
                      />
                      <div className="absolute inset-0 bg-gradient-to-t from-black via-transparent to-transparent opacity-60"></div>
                      <div className="absolute top-4 right-4 px-3 py-1 bg-[#81d8d0] text-black text-sm rounded-full">
                        #171
                      </div>
                    </div>

                    {/* Claim Form */}
                    <div className="space-y-4">
                      <h3 className="text-2xl text-white mb-4">Claim Your NFT</h3>
                      
                      <div>
                        <label className="block text-sm text-gray-400 mb-2">Wallet Address</label>
                        <div className="relative">
                          <input
                            type="text"
                            value={walletAddress}
                            onChange={(e) => setWalletAddress(e.target.value)}
                            placeholder="0x..."
                            className="w-full px-4 py-3 bg-black/50 border border-[#81d8d0]/20 rounded-xl focus:outline-none focus:border-[#81d8d0] transition-colors text-white placeholder-gray-600"
                          />
                          {walletAddress && (
                            <button className="absolute right-3 top-1/2 -translate-y-1/2 text-[#81d8d0] hover:text-white transition-colors">
                              <Copy className="w-4 h-4" />
                            </button>
                          )}
                        </div>
                      </div>

                      <motion.button
                        disabled={!walletAddress}
                        className="interact-button w-full bg-gradient-to-r from-[#81d8d0] to-[#5bb9b1] hover:from-[#6ec9c1] hover:to-[#4aa8a0] text-black py-4 rounded-xl transition-all disabled:opacity-30 disabled:cursor-not-allowed flex items-center justify-center gap-2 group"
                        whileHover={{ scale: !walletAddress ? 1 : 1.02 }}
                        whileTap={{ scale: !walletAddress ? 1 : 0.98 }}
                      >
                        <span>Claim NFT</span>
                        <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
                      </motion.button>
                    </div>
                  </div>
                </motion.div>
              ) : (
                <motion.div
                  className="relative bg-gradient-to-br from-gray-900 to-black border border-[#81d8d0]/20 rounded-3xl p-12 backdrop-blur-xl text-center"
                  initial={{ scale: 0.8, opacity: 0 }}
                  animate={{ scale: 1, opacity: 1 }}
                  transition={{ type: "spring" }}
                >
                  <motion.div
                    className="w-24 h-24 bg-gradient-to-br from-[#81d8d0] to-[#5bb9b1] rounded-full flex items-center justify-center mx-auto mb-6"
                    animate={{ rotate: 360, scale: [1, 1.1, 1] }}
                    transition={{ duration: 1 }}
                  >
                    <Gift className="w-12 h-12 text-black" />
                  </motion.div>
                  
                  <h3 className="text-3xl text-white mb-4">NFT Claimed!</h3>
                  <p className="text-gray-400 mb-6">
                    Your exclusive Tiffany & Co. NFT has been sent to your wallet
                  </p>
                  
                  <div className="bg-black/50 border border-[#81d8d0]/20 rounded-xl p-4 mb-6">
                    <p className="text-sm text-gray-500 mb-2">Transaction Hash</p>
                    <p className="text-xs text-[#81d8d0] font-mono break-all">
                      0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb7
                    </p>
                  </div>
                  
                  <motion.button
                    className="text-[#81d8d0] hover:text-white transition-colors flex items-center gap-2 mx-auto"
                    whileHover={{ x: 5 }}
                  >
                    View on OpenSea
                    <ArrowRight className="w-4 h-4" />
                  </motion.button>
                </motion.div>
              )}
            </motion.div>
          </div>
        </div>

        {/* Footer */}
        <motion.footer
          className="container mx-auto px-6 py-8 border-t border-white/5"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ delay: 1 }}
        >
          <div className="flex flex-col md:flex-row justify-between items-center gap-4 text-sm text-gray-600">
            <p>© 2025 Tiffany & Co. All rights reserved.</p>
          </div>
        </motion.footer>
      </div>
    </div>
  );
}