import { useState } from 'react';
import { motion, AnimatePresence } from 'motion/react';
import { X, Wallet, CheckCircle2, AlertCircle, Loader2 } from 'lucide-react';
import { Button } from '@/app/components/ui/button';

interface ClaimModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export function ClaimModal({ isOpen, onClose }: ClaimModalProps) {
  const [step, setStep] = useState<'connect' | 'checking' | 'eligible' | 'claiming' | 'success'>('connect');

  const handleConnect = () => {
    setStep('checking');
    setTimeout(() => setStep('eligible'), 2000);
  };

  const handleClaim = () => {
    setStep('claiming');
    setTimeout(() => setStep('success'), 3000);
  };

  return (
    <AnimatePresence>
      {isOpen && (
        <>
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            onClick={onClose}
            className="fixed inset-0 bg-black/80 backdrop-blur-sm z-50"
          />
          <div className="fixed inset-0 flex items-center justify-center z-50 p-4">
            <motion.div
              initial={{ opacity: 0, scale: 0.9, y: 20 }}
              animate={{ opacity: 1, scale: 1, y: 0 }}
              exit={{ opacity: 0, scale: 0.9, y: 20 }}
              className="bg-zinc-900 border border-zinc-800 rounded-2xl max-w-md w-full p-8 relative"
            >
              <button
                onClick={onClose}
                className="absolute top-4 right-4 text-zinc-500 hover:text-white transition-colors"
              >
                <X className="w-5 h-5" />
              </button>

              {step === 'connect' && (
                <div className="text-center">
                  <div className="w-16 h-16 bg-[#0abab5]/10 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Wallet className="w-8 h-8 text-[#0abab5]" />
                  </div>
                  <h2 className="text-2xl font-bold text-white mb-2">Connect Wallet</h2>
                  <p className="text-zinc-400 mb-6">
                    Connect your wallet to check eligibility and claim your Tiffany NFT
                  </p>
                  <Button
                    onClick={handleConnect}
                    className="w-full bg-[#0abab5] hover:bg-[#0a9a95] text-white h-12"
                  >
                    Connect MetaMask
                  </Button>
                </div>
              )}

              {step === 'checking' && (
                <div className="text-center py-8">
                  <Loader2 className="w-12 h-12 text-[#0abab5] animate-spin mx-auto mb-4" />
                  <h2 className="text-xl font-bold text-white mb-2">Checking Eligibility</h2>
                  <p className="text-zinc-400">Please wait while we verify your wallet...</p>
                </div>
              )}

              {step === 'eligible' && (
                <div className="text-center">
                  <div className="w-16 h-16 bg-green-500/10 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle2 className="w-8 h-8 text-green-500" />
                  </div>
                  <h2 className="text-2xl font-bold text-white mb-2">You're Eligible!</h2>
                  <p className="text-zinc-400 mb-6">
                    Congratulations! Your wallet qualifies for 1 Tiffany NFT.
                  </p>
                  <div className="bg-zinc-800/50 border border-zinc-700 rounded-xl p-4 mb-6">
                    <div className="flex items-center justify-between text-sm mb-2">
                      <span className="text-zinc-500">NFTs Available</span>
                      <span className="text-white font-semibold">1</span>
                    </div>
                    <div className="flex items-center justify-between text-sm">
                      <span className="text-zinc-500">Gas Fee (Est.)</span>
                      <span className="text-white font-semibold">~$15</span>
                    </div>
                  </div>
                  <Button
                    onClick={handleClaim}
                    className="w-full bg-[#0abab5] hover:bg-[#0a9a95] text-white h-12"
                  >
                    Claim NFT
                  </Button>
                </div>
              )}

              {step === 'claiming' && (
                <div className="text-center py-8">
                  <Loader2 className="w-12 h-12 text-[#0abab5] animate-spin mx-auto mb-4" />
                  <h2 className="text-xl font-bold text-white mb-2">Processing Transaction</h2>
                  <p className="text-zinc-400">Please confirm the transaction in your wallet...</p>
                </div>
              )}

              {step === 'success' && (
                <div className="text-center">
                  <motion.div
                    initial={{ scale: 0 }}
                    animate={{ scale: 1 }}
                    transition={{ type: 'spring', duration: 0.6 }}
                    className="w-16 h-16 bg-green-500/10 rounded-full flex items-center justify-center mx-auto mb-4"
                  >
                    <CheckCircle2 className="w-8 h-8 text-green-500" />
                  </motion.div>
                  <h2 className="text-2xl font-bold text-white mb-2">NFT Claimed!</h2>
                  <p className="text-zinc-400 mb-6">
                    Your Tiffany NFT has been successfully minted to your wallet.
                  </p>
                  <div className="bg-zinc-800/50 border border-zinc-700 rounded-xl p-4 mb-6">
                    <div className="text-xs text-zinc-500 mb-1">Transaction Hash</div>
                    <div className="text-xs text-white font-mono break-all">
                      0x7f9fade1c0d57a7af66ab4ead79fade1c0d57a7af66ab4ead7c2c2eb7b11a91385
                    </div>
                  </div>
                  <Button
                    onClick={onClose}
                    variant="outline"
                    className="w-full border-zinc-700 text-white hover:bg-zinc-800"
                  >
                    Close
                  </Button>
                </div>
              )}
            </motion.div>
          </div>
        </>
      )}
    </AnimatePresence>
  );
}
